<?php

class AIKIT_Comments extends AIKIT_Page {

    const TABLE_NAME_COMMENT_JOBS = 'aikit_comment_jobs';

    const AIKIT_GENERATED_COMMENT = 'aikit_generated_comment';
    private static $instance = null;

    // singleton
    public static function get_instance()
    {
        if (self::$instance == null) {
            self::$instance = new AIKIT_Comments();
        }
        return self::$instance;
    }

    public function __construct()
    {
        add_action( 'rest_api_init', function () {

            register_rest_route( 'aikit/comments/v1', '/delete', array(
                'methods' => 'POST',
                'callback' => array($this, 'delete_job'),
                'permission_callback' => function () {
                    return is_user_logged_in() && current_user_can( 'edit_posts' );
                }
            ));

            register_rest_route( 'aikit/comments/v1', '/delete-all', array(
                'methods' => 'POST',
                'callback' => array($this, 'delete_all_jobs'),
                'permission_callback' => function () {
                    return is_user_logged_in() && current_user_can( 'edit_posts' );
                }
            ));

        });


        add_action('aikit_comments', array($this, 'execute'));

        add_action('add_meta_boxes', array($this, 'add_comments_meta_box'));
        add_action('save_post', array($this, 'save_comments_meta_box'));
    }

    public function delete_job($data)
    {
        $id = $data['id'] ?? null;

        if (empty($id)) {
            return new WP_REST_Response(['success' => false], 400);
        }

        $job = $this->get_job_by_id($id);

        if (!$job) {
            return new WP_REST_Response(['success' => false], 404);
        }

        global $wpdb;
        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;

        $wpdb->delete(
            $table_name,
            array('id' => $id)
        );

        return new WP_REST_Response(['success' => true], 200);
    }

    public function delete_all_jobs($data){
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;

        $wpdb->query("DELETE FROM $table_name WHERE status = 'pending'");

        return new WP_REST_Response(array(
            'success' => true,
        ));
    }

    public function execute()
    {
        global $wpdb;
        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;

        $jobs = $wpdb->get_results("SELECT * FROM $table_name WHERE status = 'pending' AND is_running = 0 ORDER BY date_created ASC LIMIT 2");

        foreach ($jobs as $job) {

            $wpdb->update(
                $table_name,
                array(
                    'is_running' => 1,
                    'date_modified' => current_time('mysql', true),
                ),
                array(
                    'id' => $job->id,
                )
            );

            $generated_comments = [];

            $faker_locale = $this->map_language_to_locale($job->language);
            $faker = AIKit\Dependencies\Faker\Factory::create($faker_locale);

            try {

                $post_content = get_post_field('post_content', $job->post_id);

                $date = $faker->dateTimeBetween('-3 hour', '-2 hour');

                for ($i = 0; $i < $job->count; $i++) {
                    $paragraph = $this->get_paragraph_from_post_content($post_content);

                    $prompt = str_replace('[[text]]', $paragraph, $job->prompt);

                    // random between 100 and 300
                    $max_tokens = rand(50, 100);

                    $comment = aikit_ai_text_generation_request($prompt, $max_tokens);

                    $name = rand(0, 1) == 0 ? $faker->firstNameMale() . ' ' . $faker->lastName() : $faker->firstNameFemale() . ' ' . $faker->lastName();
                    $name = rand(0, 1) == 0 ? strtolower($name) : $name;

                    // add comments to post
                    $comment_id = wp_insert_comment([
                        'comment_post_ID' => $job->post_id,
                        'comment_author' => $name,
                        'comment_author_email' => $faker->email(),
                        'comment_content' => $comment,
                        'comment_type' => 'comment',
                        'comment_approved' => 1,
                        'comment_date' => $date->format('Y-m-d H:i:s'),
                        'comment_date_gmt' => $date->format('Y-m-d H:i:s'),
                    ]);

                    // add few minutes to the date
                    $date->modify('+'.rand(1, 10).' minutes');

                    // add a comment meta to indicate that this comment was generated by AIKit
                    add_comment_meta($comment_id, self::AIKIT_GENERATED_COMMENT, true);

                    $generated_comments[] = $comment;
                }

            } catch (Throwable $e) {
                $logs = json_decode($job->logs);
                $logs[] = array(
                    'message' => $e->getMessage(),
                    'date' => current_time( 'mysql', true )
                );

                $error_count = intval($job->error_count);
                $error_count++;

                $wpdb->update(
                    $table_name,
                    array(
                        'is_running' => 0,
                        'date_modified' => current_time('mysql', true),
                        'logs' => json_encode($logs),
                        'error_count' => $error_count,
                    ),
                    array(
                        'id' => $job->id,
                    )
                );

                continue;
            }

            $wpdb->update(
                $table_name,
                array(
                    'generated_comments' => json_encode($generated_comments),
                    'is_running' => 0,
                    'date_modified' => current_time('mysql', true),
                    'status' => 'completed',
                ),
                array(
                    'id' => $job->id,
                )
            );
        }
    }

    public function get_paragraph_from_post_content($post_content)
    {
        // add a new line at the end of every html end tag
        $post_content = preg_replace('/(<\/[^>]+>)/', "$1\n", $post_content);

        //strip html tags
        $post_content = strip_tags($post_content);

        $paragraphs = explode("\n", $post_content);

        $paragraphs = array_filter($paragraphs, function ($paragraph) {
            return !empty(trim($paragraph));
        });

        $paragraphs = array_values($paragraphs);

        // combine paragraphs that are too short
        $combined_paragraphs = [];
        $combined_paragraph = '';

        $tokenizer = AIKIT_Gpt3Tokenizer::getInstance();

        foreach ($paragraphs as $paragraph) {
            if ($tokenizer->count($combined_paragraph . ' ' . $paragraph) < 1000) {
                $combined_paragraph .= ' ' . $paragraph . "\n";
            } else {
                if (!empty($combined_paragraph)) {
                    $combined_paragraphs[] = $combined_paragraph;
                }
                $combined_paragraph = $paragraph . "\n";
            }
        }

        if (!empty($combined_paragraph)) {
            $combined_paragraphs[] = $combined_paragraph;
        }

        return $combined_paragraphs[rand(0, count($combined_paragraphs) - 1)];
    }

    public function render()
    {
        $cron_url = get_site_url() . '/wp-cron.php';

        ?>
        <div class="wrap">
            <h1><?php echo esc_html__( 'AI Comments', 'aikit' ); ?></h1>
            <p>
                <a href="#" class="aikit-util-button float-end mx-1 aikit-top-hidden-toggle btn btn-outline-secondary btn-sm"><i class="bi bi-info-lg"></i> <?php echo esc_html__( 'How to setup?', 'aikit' ); ?></a>
                <?php echo esc_html__( 'AIKit AI comments allows you to automatically generate comments for your posts using AI to increase engagement and SEO.', 'aikit' ); ?>
            </p>

            <div class="aikit-top-hidden-note">
                <p>
                    <strong><?php echo esc_html__( 'Note:', 'aikit' ); ?></strong>
                    <?php echo esc_html__('AIKit AI comments creation jobs run in the background as scheduled jobs.', 'aikit'); ?>
                    <?php echo esc_html__( 'By default, WordPress scheduled jobs only run when someone visits your site. To ensure that your comment generation jobs run even if nobody visits your site, you can set up a cron job on your server to call the WordPress cron system at regular intervals. Please ask your host provider to do that for you. Here is the cron job definition:', 'aikit' ); ?>
                    <code>
                        */5 * * * * curl -I <?php echo $cron_url ?> >/dev/null 2>&1
                    </code>
                </p>
            </div>

            <ul class="nav nav-tabs aikit-comments-tabs">
                <li class="nav-item">
                    <a class="nav-link active" href="<?php echo admin_url( 'admin.php?page=aikit_comments&action=jobs' ); ?>"><?php echo esc_html__( 'Jobs', 'aikit' ); ?></a>
                </li>
            </ul>

            <div class="aikit-comments-content">
                <?php
                if (isset($_GET['job_id']))
                    $this->render_view_tab($_GET['job_id']);
                else {
                    $this->render_listing_tab();
                }
                ?>
            </div>

        </div>

        <?php
    }

    public function render_view_tab($job_id)
    {
        $job = $this->get_job_by_id($job_id);

        if (empty($job)) {
            ?>
            <div class="notice notice-error">
                <p>
                    <?php echo esc_html__( 'Job not found.', 'aikit' ); ?>
                </p>
            </div>
            <?php

            return;
        }

        $status_message = '<i class="bi bi-hourglass-split"></i> ' . esc_html__( 'Please set back and relax while the your comment generation job is being processed. Status of the job will change to reflect when this is done.', 'aikit');

        if ($job->status === 'completed') {
            $status_message = '<i class="bi bi-check-circle-fill"></i> ' . esc_html__( 'Congratulations! Your text to comment generation job is processed successfully and the generated comments have been added to your post.', 'aikit');
        }

        ?>
        <p class="aikit-notice-message">

            <?php
            echo $status_message;
            ?>
        </p>
        <?php

        $this->render_job_info_tabs($job);
    }

    private function render_job_info_tabs($job)
    {
        $logs = json_decode($job->logs, true);

        if (empty($logs)) {
            $logs = [];
        }

        $logsCount = count($logs) == 0 ? '' : ' (' . count($logs) . ')';
        ?>
        <ul class="nav nav-tabs aikit-embeddings-job-info-tabs mt-5">
            <li class="nav-item">
                <a class="nav-link active" href="#job-info" data-bs-toggle="tab"><?php echo esc_html__( 'Job Info', 'aikit' ); ?></a>
            </li>
            <li class="nav-item">
                <a class="nav-link" href="#logs" data-bs-toggle="tab"><?php echo esc_html__( 'Logs', 'aikit' ) . $logsCount; ?></a>
            </li>
        </ul>

        <div class="tab-content">
            <div class="tab-pane fade show active" id="job-info">

                <div class="row mt-2">
                    <div class="col">
                        <span class=""><?php echo __('Status', 'aikit'); ?></span>: <span class="badge badge-pill badge-dark aikit-badge-active ms-1"><?php echo $this->map_comments_job_status($job->status) ?></span>
                    </div>
                </div>

                <div class="row mt-2">
                    <div class="col">
                        <span class=""><?php echo __('Comments to generate', 'aikit'); ?></span>: <span class="badge badge-pill badge-dark aikit-badge-active ms-1"><?php echo $job->count ?></span>
                    </div>
                </div>

                <div class="row mt-2">
                    <div class="col">
                        <span class=""><?php echo __('Post', 'aikit'); ?></span>: <a href="<?php echo get_edit_post_link($job->post_id); ?>" target="_blank"><?php echo get_the_title($job->post_id); ?></a>
                    </div>
                </div>

                <div class="row mt-2">
                    <div class="col">
                        <span class=""><?php echo __('Prompt', 'aikit'); ?></span>: <textarea readonly rows="3" style="width: 100%; margin-top:5px"><?php echo $job->prompt ?></textarea>
                    </div>
                </div>


                <?php if ($job->generated_comments) { ?>
                <div class="row mt-2">
                    <div class="col">
                        <span class=""><?php echo __('Generated comments', 'aikit'); ?></span>:
                        <?php
                        $generated_comments = json_decode($job->generated_comments, true);

                        foreach ($generated_comments as $comment) {
                            ?>
                            <textarea readonly rows="3" style="width: 100%; margin-top:5px"><?php echo $comment ?></textarea>
                            <?php
                        }
                        ?>


                    </div>
                </div>
                <?php } ?>


            </div>
            <div class="tab-pane fade" id="logs">
                <div class="alert alert-success my-4" role="alert">
                    <?php echo esc_html__( 'API errors can happen due to many reasons (such as API being down or rate limits have been exceeded, etc). In case of errors, do not worry, AIKit will keep retrying the job till it succeeds.', 'aikit' ); ?>
                </div>

                <?php

                if (count($logs) === 0) {
                    echo esc_html__( 'No logs found.', 'aikit' );
                }

                foreach ($logs as $log) {
                    ?>
                    <pre><code class="json"><?php echo esc_html(json_encode($log, JSON_PRETTY_PRINT)); ?></code></pre>
                    <?php
                }
                ?>
            </div>

        </div>
        <?php
    }

    public function render_listing_tab()
    {
        // get all jobs from DB
        global $wpdb;
        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;

        $delete_all_url = get_site_url() . '/?rest_route=/aikit/comments/v1/delete-all';

        $total_jobs = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");

        if ($total_jobs > 0) {
            ?>
            <div class="row mb-2 float-end">
                <div class="col">
                    <button data-confirm-message="<?php echo esc_html__('Are you sure you want to delete all pending jobs?', 'aikit') ?>" id="aikit-comments-delete-all" class="btn btn-sm btn-outline-danger ms-2" type="button" href="<?php echo $delete_all_url ?>"><i class="bi bi-trash3-fill"></i> <?php echo esc_html__( 'Delete all pending jobs', 'aikit' ); ?></button>
                </div>
            </div>
            <?php
        }

        $paged = isset($_GET['paged']) ? intval($_GET['paged']) : 1;
        $per_page = 25;
        $columns = [
            esc_html__('Post title', 'aikit'),
            esc_html__('Status', 'aikit'),
            esc_html__('Date created', 'aikit'),
            esc_html__('Comment count to generate', 'aikit'),
            esc_html__('Actions', 'aikit'),
        ];
        $html = '<table class="table" id="aikit-comments-jobs">
            <thead>
            <tr>';

        foreach ($columns as $column) {
            $html .= '<th scope="col">' . $column . '</th>';
        }

        $html .= '
            </tr>
            </thead>
            <tbody>';

        // prepared statement to prevent SQL injection with pagination
        $jobs = $wpdb->get_results($wpdb->prepare("SELECT * FROM $table_name ORDER BY id DESC LIMIT %d, %d", ($paged - 1) * $per_page, $per_page));

        if (empty($jobs)) {
            $html .= '<tr>
                <td colspan="' . count($columns) . '">' . esc_html__('No entries found', 'aikit') . '</td>
            </tr>';
        }

        $page_url = get_admin_url() . 'admin.php?page=aikit_comments&action=view';

        foreach ($jobs as $job) {
            $current_page_url = $page_url . '&job_id=' . $job->id;
            $title = get_the_title($job->post_id);
            if (empty($title)) {
                $title = esc_html__('(no title)', 'aikit');
            }

            $delete_url = get_site_url() . '/?rest_route=/aikit/comments/v1/delete';

            // link to edit post on admin
            $editPostUrl = get_edit_post_link($job->post_id);
            $html .= '<tr>
                <td>' . '<a href="' . $current_page_url . '">' . $title . '</a> <a class="aikit-view-post-link" href="' . $editPostUrl . '" target="_blank"><i class="bi bi-box-arrow-up-right"></i> ' . '</a>' . '</td>
                <td><span class="badge badge-pill badge-dark aikit-badge-active">' . $this->map_comments_job_status($job->status) . '</span></td>
                <td>' . (empty($job->date_created) ? '-' : aikit_date($job->date_created)) . '</td>       
                <td>' . $job->count . '</td>        
                <td>
                    <a href="' . $page_url . '&job_id=' . $job->id . '" title="' . __('View', 'aikit') . '" class="aikit-comments-action" data-id="' . $job->id . '"><i class="bi bi-eye-fill"></i></a>
                    <a href="' . $delete_url . '" title="' . __('Delete', 'aikit') . '" class="aikit-comments-job-delete aikit-comments-action" data-confirm-message="' . __('Are you sure you want to delete this comments jobs?', 'aikit') . '" data-id="' . $job->id . '"><i class="bi bi-trash-fill"></i></a>                  
                </td>
            </tr>';
        }

        // pagination
        $total_jobs = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
        $total_pages = ceil($total_jobs / $per_page);

        if ($total_pages > 1) {
            $html .= '<tr>
            <td colspan="' . count($columns) . '">';

            // previous page

            $html .= '<div class="aikit-jobs-pagination">';
            if ($paged > 1) {
                $html .= '<a href="' . $page_url . '&paged=' . ($paged - 1) . '">' . __('« Previous', 'aikit') . '</a>';
            }

            for ($i = 1; $i <= $total_pages; $i++) {
                // add class to current page
                $current_page_class = '';
                if ($paged == $i) {
                    $current_page_class = 'aikit-jobs-pagination-current';
                }

                $html .= '<a class="' . $current_page_class . '" href="' . $page_url . '&paged=' . $i . '" data-page="' . $i . '">' . $i . '</a>';
            }

            // next page
            if ($paged < $total_pages) {
                $html .= '<a href="' . $page_url . '&paged=' . ($paged + 1) . '">' . __('Next »', 'aikit') . '</a>';
            }

            $html .= '</div>';

            $html .= '</td>
            </tr>';
        }

        $html .= '</tbody>
        
        </table>';

        echo $html;
    }

    function map_language_to_locale ($language) {
        $map = [
            'ar' => 'ar_EG',
            'da' => 'da_DK',
            'el' => 'el_GR',
            'en' => 'en_US',
            'et' => 'et_EE',
            'fa' => 'fa_IR',
            'he' => 'he_IL',
            'hy' => 'hy_AM',
            'ka' => 'ka_GE',
            'kk' => 'kk_KZ',
            'ko' => 'ko_KR',
            'ms' => 'ms_MY',
            'sv' => 'sv_SE',
            'uk' => 'uk_UA',
            'vi' => 'vi_VN',
            'zh' => 'zh_CN',
        ];

        if (isset($map[$language])) {
            return $map[$language];
        }

        return $language.'_'.strtoupper($language);
    }

    public function activate_scheduler()
    {
        if (! wp_next_scheduled ( 'aikit_comments')) {
            wp_schedule_event( time(), 'every_10_minutes', 'aikit_comments');
        }
    }

    public function deactivate_scheduler()
    {
        wp_clear_scheduled_hook('aikit_comments');
    }

    public function do_db_migration()
    {
        global $wpdb;

        $charset_collate = $wpdb->get_charset_collate();

        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            status varchar(255) NOT NULL,
            post_id int(11) NOT NULL,
            prompt LONGTEXT NOT NULL,
            count int(11) NOT NULL,
            language varchar(255) NOT NULL,
            generated_comments LONGTEXT NULL,
            date_created datetime DEFAULT NULL,
            date_modified datetime DEFAULT NULL,
            logs LONGTEXT NULL,
            is_running BOOLEAN DEFAULT FALSE,
            error_count INT DEFAULT 0,
            INDEX aikit_cmnt_unq_post_id (post_id),
            PRIMARY KEY  (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta( $sql );
    }

    function add_comments_meta_box() {
        add_meta_box(
            'aikit_comments_meta_box',
            __('AIKit AI Comments', 'aikit'),
            array($this, 'render_comments_meta_box'),
            ['post', 'page'],
            'side', // This places the box in the sidebar
            'core'
        );
    }

    private function map_comments_job_status($status)
    {
        return [
            'pending' => esc_html__('Pending', 'aikit'),
            'completed' => esc_html__('Completed', 'aikit'),
        ][strtolower($status)];
    }

    function get_job_by_id($job_id)
    {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;

        $job = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE id = %d", $job_id));

        return $job;
    }

    function get_jobs_for_post($post_id)
    {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;

        $sql = "SELECT * FROM $table_name WHERE post_id = $post_id";

        $jobs = $wpdb->get_results($sql);

        return $jobs;
    }

    function add_job_for_post($post_id, $comment_count_option, $comment_prompt)
    {
        global $wpdb;

        $table_name = $wpdb->prefix . self::TABLE_NAME_COMMENT_JOBS;

        $wpdb->insert(
            $table_name,
            [
                'status' => 'pending',
                'post_id' => $post_id,
                'prompt' => $comment_prompt,
                'language' => aikit_get_language_used(),
                'count' => $comment_count_option,
                'date_created' => current_time('mysql', true),
                'date_modified' => current_time('mysql', true),
            ]
        );
    }

    function render_comments_meta_box($post) {

        $jobs = $this->get_jobs_for_post($post->ID);

        if (count($jobs) > 0) {
            echo esc_html__('Job(s) already exist for this post:', 'aikit');

            ?>
                <div style="display: flex; gap: 5px; flex-wrap: wrap; margin-top: 10px; margin-bottom: 10px;">
            <?php
            foreach ($jobs as $job) {
                $link = get_admin_url() . 'admin.php?page=aikit_comments&action=view&job_id=' . $job->id;
                ?>
                <a style="text-decoration: none; background-color: #0c63e4; border-radius: 999px; color: white; padding: 7px; font-size: 10px;" href="<?php echo $link; ?>"><?php echo $this->map_comments_job_status($job->status); ?></a>
                <?php
            }
            ?>
                </div>
            <?php
        }

        $saved_prompt = get_option('aikit_comments_prompt');
        $prompt = !empty($saved_prompt) ? $saved_prompt : AIKIT_COMMENTS_PROMPTS[aikit_get_language_used()]['prompt'] ?? AIKIT_COMMENTS_PROMPTS['en']['prompt'];

        // display number of comments to be generated, the prompt to be used and the generate comments and the current "language selected" pill.
        $selected_language = aikit_get_language_used();
        $languages = AIKit_Admin::instance()->get_languages();
        $selected_language_name = $languages[$selected_language]['name'] ?? 'English';

        ?>
        <label for="aikit_generate_comments_option">
            <input type="checkbox" name="aikit_generate_comments_option" id="aikit_generate_comments_option" value="1">
            <b><?php echo count($jobs) > 0 ? esc_html__('Generate more AI comments', 'aikit') : esc_html__('Generate AI comments', 'aikit'); ?></b>
        </label>
        <br>
        <div style="margin-top: 8px;">
            <label for="aikit_comment_count_option" >
                <?php echo esc_html__('Number of comments to generate', 'aikit'); ?>
                <br>
                <input type="number" name="aikit_comment_count_option" id="aikit_comment_count_option" value="3" size="2" style="width: 70px; margin-left:5px; margin-top:5px">
            </label>
        </div>
        <div style="margin-top: 8px;">
            <label for="aikit_comment_prompt" >
                <?php echo esc_html__('Prompt', 'aikit'); ?>
            </label>
            <br>
            <textarea name="aikit_comment_prompt" id="aikit_comment_prompt" rows="3" style="width: 100%; margin-top:5px"><?php echo $prompt ?></textarea>
        </div>
        <div style="margin-top: 8px;">
            <span>
                <?php echo esc_html__('Selected Language', 'aikit'); ?>
                <code>
                    <?php echo $selected_language_name; ?>
                </code>
            </span>
            <br>
            <a href="<?php echo admin_url( 'admin.php?page=aikit' ); ?>" style="font-size: 11px;">
                <?php echo esc_html__('Change Language', 'aikit'); ?>
            </a>
        </div>
        <br>
        <label for="aikit_save_prompt_option">
            <input type="checkbox" name="aikit_save_prompt_option" id="aikit_save_prompt_option" value="1">
            <b><?php echo esc_html__('Save prompt for future use', 'aikit'); ?></b>
        </label>
        <br>
        <br>
        <label for="aikit_reset_prompt_option">
            <input type="checkbox" name="aikit_reset_prompt_option" id="aikit_reset_prompt_option" value="1">
            <b><?php echo esc_html__('Reset prompt to default', 'aikit'); ?></b>
        </label>
        <br>
        <br>

        <p style="font-size: 12px">
            <?php echo __('AIKit AI Comments generates AI comments for your post.', 'aikit'); ?>
            <br/>
            <br/>
            <?php echo __('Make sure to include the ', 'aikit'); ?>
            <code>[[text]]</code>
            <?php echo __(' phrase in your prompt as this will be replaced with the post content before doing the AI reneration request.', 'aikit'); ?>
        </p>
        <?php
    }

    function save_comments_meta_box($post_id)
    {
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;

        if (wp_is_post_revision($post_id)) return;

        $generate_comments_option = isset($_POST['aikit_generate_comments_option']) ? $_POST['aikit_generate_comments_option'] : null;
        $save_prompt_for_future_use = isset($_POST['aikit_save_prompt_option']) ? $_POST['aikit_save_prompt_option'] : null;
        $reset_prompt = isset($_POST['aikit_reset_prompt_option']) ? $_POST['aikit_reset_prompt_option'] : null;
        $comment_count_option = isset($_POST['aikit_comment_count_option']) ? $_POST['aikit_comment_count_option'] : null;
        $comment_prompt = isset($_POST['aikit_comment_prompt']) ? $_POST['aikit_comment_prompt'] : null;

        if ($reset_prompt) {
            delete_option('aikit_comments_prompt');
            $comment_prompt = AIKIT_COMMENTS_PROMPTS[aikit_get_language_used()]['prompt'] ?? AIKIT_COMMENTS_PROMPTS['en']['prompt'];
        } else if ($save_prompt_for_future_use) {
            update_option('aikit_comments_prompt', $comment_prompt);
        }

        if ($generate_comments_option) {
            $this->add_job_for_post($post_id, $comment_count_option, $comment_prompt);
        }
    }

    public function enqueue_scripts($hook)
    {
        if ( 'aikit_page_aikit_comments' !== $hook ) {
            return;
        }

        $version = aikit_get_plugin_version();
        if ($version === false) {
            $version = rand( 1, 10000000 );
        }

        wp_enqueue_style( 'aikit_bootstrap_css', plugins_url( '../../css/bootstrap.min.css', __FILE__ ), array(), $version );
        wp_enqueue_style( 'aikit_bootstrap_icons_css', plugins_url( '../../css/bootstrap-icons.css', __FILE__ ), array(), $version );
        wp_enqueue_style( 'aikit_comments_css', plugins_url( '../../css/comments.css', __FILE__ ), array(), $version );

        wp_enqueue_script( 'aikit_bootstrap_js', plugins_url('../../js/bootstrap.bundle.min.js', __FILE__ ), array(), $version );
        wp_enqueue_script( 'aikit_jquery_ui_js', plugins_url('../../js/jquery-ui.min.js', __FILE__ ), array('jquery'), $version );
        wp_enqueue_script( 'aikit_comments_js', plugins_url( '../../js/comments.js', __FILE__ ), array( 'jquery' ), array(), $version );
    }
}